<?php
/**
 * Plugin Name: Sistemsel Click Tracker
 * Plugin URI: https://plugin.sistemsel.com/updates/sistemsel-click-tracker/
 * Description: Belirlenen CSS ID’lere tıklamaları kaydeder, günlük raporlar ve log temizleme sunar.
 * Version: 26.01.14
 * Author: Sistemsel Medya
 * Author URI: https://sistemsel.com
 * License: GPL2+
 * Text Domain: sistemsel-click-tracker
 */


if (!defined('ABSPATH')) exit;

class Sistemsel_Click_Tracker {
  const OPTION_KEY = 'sct_tracked_elements';
  const NONCE_ACTION = 'sct_nonce_action';

  public function __construct() {
    register_activation_hook(__FILE__, [$this, 'activate']);
    add_action('admin_menu', [$this, 'admin_menu']);
    add_action('admin_init', [$this, 'register_settings']);

    add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);

    add_action('wp_ajax_sct_click', [$this, 'ajax_click']);
    add_action('wp_ajax_nopriv_sct_click', [$this, 'ajax_click']);
  }

  public function table_name() {
    global $wpdb;
    return $wpdb->prefix . 'sct_clicks';
  }

  public function activate() {
    global $wpdb;

    $table = $this->table_name();
    $charset = $wpdb->get_charset_collate();

    require_once ABSPATH . 'wp-admin/includes/upgrade.php';

    // clicks tablosu: ham tıklama kaydı
    $sql = "CREATE TABLE $table (
      id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
      clicked_at DATETIME NOT NULL,
      element_id VARCHAR(190) NOT NULL,
      element_name VARCHAR(190) NULL,
      page_url TEXT NULL,
      referrer TEXT NULL,
      ip_hash CHAR(64) NULL,
      user_id BIGINT UNSIGNED NULL,
      PRIMARY KEY (id),
      KEY clicked_at (clicked_at),
      KEY element_id (element_id)
    ) $charset;";

    dbDelta($sql);

    // Varsayılan ayar
    if (get_option(self::OPTION_KEY) === false) {
      update_option(self::OPTION_KEY, [
        ['id' => 'btnTeklifAl', 'name' => 'Teklif Al'],
        ['id' => 'btnWhatsapp', 'name' => 'WhatsApp']
      ]);
    }
  }

  public function admin_menu() {
    add_menu_page(
      'Click Tracker',
      'Click Tracker',
      'manage_options',
      'sct-click-tracker',
      [$this, 'admin_page'],
      'dashicons-chart-bar',
      58
    );
  }

  public function register_settings() {
    register_setting('sct_settings_group', self::OPTION_KEY, [
      'type' => 'array',
      'sanitize_callback' => [$this, 'sanitize_elements'],
      'default' => []
    ]);
  }

  public function sanitize_elements($value) {
    $clean = [];
    if (!is_array($value)) return $clean;

    foreach ($value as $row) {
      if (!is_array($row)) continue;

      $id = isset($row['id']) ? sanitize_text_field($row['id']) : '';
      $name = isset($row['name']) ? sanitize_text_field($row['name']) : '';

      if ($id === '') continue;

      // CSS ID'de # olmadan saklıyoruz. (btnTeklifAl gibi)
      $id = ltrim($id, '#');

      $clean[] = [
        'id' => $id,
        'name' => $name,
      ];
    }
    return $clean;
  }

  public function enqueue_scripts() {
    $elements = get_option(self::OPTION_KEY, []);
    if (!is_array($elements) || empty($elements)) return;

    $map = [];
    foreach ($elements as $el) {
      if (!empty($el['id'])) {
        $map[$el['id']] = isset($el['name']) ? $el['name'] : '';
      }
    }

    wp_enqueue_script(
      'sct-tracker',
      plugin_dir_url(__FILE__) . 'assets/tracker.js',
      [],
      '1.0.0',
      true
    );

    wp_localize_script('sct-tracker', 'SCT', [
      'ajaxurl' => admin_url('admin-ajax.php'),
      'nonce' => wp_create_nonce(self::NONCE_ACTION),
      'elements' => $map
    ]);
  }

  public function ajax_click() {
    $nonce = isset($_POST['nonce']) ? sanitize_text_field($_POST['nonce']) : '';
    if (!wp_verify_nonce($nonce, self::NONCE_ACTION)) {
      wp_send_json_error(['message' => 'Invalid nonce'], 403);
    }

    $element_id = isset($_POST['element_id']) ? sanitize_text_field($_POST['element_id']) : '';
    $page_url   = isset($_POST['page_url']) ? esc_url_raw($_POST['page_url']) : '';
    $referrer   = isset($_POST['referrer']) ? esc_url_raw($_POST['referrer']) : '';

    if ($element_id === '') {
      wp_send_json_error(['message' => 'Missing element_id'], 400);
    }

    // sadece tanımlı elementler kaydedilsin
    $elements = get_option(self::OPTION_KEY, []);
    $name = '';
    $allowed = false;

    foreach ((array)$elements as $el) {
      if (!empty($el['id']) && $el['id'] === $element_id) {
        $allowed = true;
        $name = isset($el['name']) ? $el['name'] : '';
        break;
      }
    }

    if (!$allowed) {
      wp_send_json_success(['message' => 'Ignored (not configured)']);
    }

    global $wpdb;
    $table = $this->table_name();

    $ip = isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field($_SERVER['REMOTE_ADDR']) : '';
    $ip_hash = $ip ? hash('sha256', $ip . NONCE_SALT) : null;

    $user_id = get_current_user_id();
    $user_id = $user_id ? $user_id : null;

    $wpdb->insert($table, [
      'clicked_at'   => current_time('mysql'),
      'element_id'   => $element_id,
      'element_name' => $name,
      'page_url'     => $page_url,
      'referrer'     => $referrer,
      'ip_hash'      => $ip_hash,
      'user_id'      => $user_id
    ], [
      '%s','%s','%s','%s','%s','%s','%d'
    ]);

    wp_send_json_success(['message' => 'Logged']);
  }

  public function admin_page() {
    if (!current_user_can('manage_options')) return;

    // Temizleme işlemi
    if (isset($_POST['sct_clear_logs'])) {
      check_admin_referer('sct_clear_logs_action');

      global $wpdb;
      $wpdb->query("TRUNCATE TABLE " . $this->table_name());

      echo '<div class="notice notice-success"><p>Loglar temizlendi.</p></div>';
    }

    $elements = get_option(self::OPTION_KEY, []);

    // Rapor filtreleri
    $from = isset($_GET['from']) ? sanitize_text_field($_GET['from']) : '';
    $to   = isset($_GET['to']) ? sanitize_text_field($_GET['to']) : '';

    // default: son 7 gün
    if ($from === '' || $to === '') {
      $to = wp_date('Y-m-d');
      $from = wp_date('Y-m-d', strtotime('-6 days'));
    }

    global $wpdb;
    $table = $this->table_name();

    // Günlük özet rapor
    $sql = $wpdb->prepare("
      SELECT DATE(clicked_at) as day,
             element_id,
             MAX(element_name) as element_name,
             COUNT(*) as clicks
      FROM $table
      WHERE clicked_at >= %s AND clicked_at <= %s
      GROUP BY DATE(clicked_at), element_id
      ORDER BY day DESC, clicks DESC
    ", $from . " 00:00:00", $to . " 23:59:59");

    $rows = $wpdb->get_results($sql);
    ?>
    <div class="wrap">
      <h1>Sistemsel Click Tracker</h1>

      <h2>Takip Edilen Elementler</h2>
      <form method="post" action="options.php">
        <?php settings_fields('sct_settings_group'); ?>

        <p>CSS ID’leri <code>#</code> olmadan gir (örn: <code>btnTeklifAl</code>). İsim, raporda görünen başlıktır.</p>

        <table class="widefat striped">
          <thead>
            <tr>
              <th style="width: 260px;">CSS ID</th>
              <th>İsim</th>
              <th style="width:120px;">Sil</th>
            </tr>
          </thead>
          <tbody id="sct-rows">
            <?php if (!empty($elements)): ?>
              <?php foreach ($elements as $i => $el): ?>
                <tr>
                  <td><input type="text" name="<?php echo esc_attr(self::OPTION_KEY); ?>[<?php echo (int)$i; ?>][id]" value="<?php echo esc_attr($el['id']); ?>" class="regular-text" /></td>
                  <td><input type="text" name="<?php echo esc_attr(self::OPTION_KEY); ?>[<?php echo (int)$i; ?>][name]" value="<?php echo esc_attr($el['name']); ?>" class="regular-text" /></td>
                  <td><button type="button" class="button sct-remove">Kaldır</button></td>
                </tr>
              <?php endforeach; ?>
            <?php endif; ?>
          </tbody>
        </table>

        <p>
          <button type="button" class="button" id="sct-add">+ Yeni Ekle</button>
          <?php submit_button('Kaydet', 'primary', 'submit', false); ?>
        </p>
      </form>

      <hr />

      <h2>Günlük Rapor</h2>
      <form method="get">
        <input type="hidden" name="page" value="sct-click-tracker" />
        <label>Başlangıç: <input type="date" name="from" value="<?php echo esc_attr($from); ?>" /></label>
        <label style="margin-left:10px;">Bitiş: <input type="date" name="to" value="<?php echo esc_attr($to); ?>" /></label>
        <button class="button">Filtrele</button>
      </form>

      <table class="widefat striped" style="margin-top:12px;">
        <thead>
          <tr>
            <th>Gün</th>
            <th>Element</th>
            <th>CSS ID</th>
            <th>Tıklama</th>
          </tr>
        </thead>
        <tbody>
          <?php if (!empty($rows)): ?>
            <?php foreach ($rows as $r): ?>
              <tr>
                <td><?php echo esc_html($r->day); ?></td>
                <td><?php echo esc_html($r->element_name ?: '-'); ?></td>
                <td><code>#<?php echo esc_html($r->element_id); ?></code></td>
                <td><?php echo (int)$r->clicks; ?></td>
              </tr>
            <?php endforeach; ?>
          <?php else: ?>
            <tr><td colspan="4">Bu tarih aralığında kayıt yok.</td></tr>
          <?php endif; ?>
        </tbody>
      
      </table>

      <form method="get" action="<?php echo admin_url('admin-post.php'); ?>" style="margin-top:15px;">
        <input type="hidden" name="action" value="sct_export_csv" />
        <input type="hidden" name="from" value="<?php echo esc_attr($from); ?>" />
        <input type="hidden" name="to" value="<?php echo esc_attr($to); ?>" />
        <?php wp_nonce_field('sct_export_csv_action'); ?>
        <button class="button button-secondary">CSV olarak dışa aktar</button>
      </form>

      <h2 style="margin-top:20px;">Günlüğü Temizle</h2>
    
      <form method="post">
        <?php wp_nonce_field('sct_clear_logs_action'); ?>
        <p>Bu işlem tüm tıklama kayıtlarını siler.</p>
        <button type="submit" name="sct_clear_logs" class="button button-danger" onclick="return confirm('Tüm loglar silinsin mi?');">
          Logları Temizle
        </button>
      </form>

      
      <hr />
      <div style="max-width:900px;background:#fff;border:1px solid #e5e7eb;border-radius:14px;padding:20px;display:flex;gap:18px;align-items:flex-start;">
        <div style="flex:0 0 160px;text-align:center;">
          <img src="https://plugin.sistemsel.com/sistemsel/logo.png" alt="Sistemsel Medya ve Yazılım" style="max-width:140px;height:auto;margin-bottom:10px;" />
        </div>
        <div style="flex:1;">
          <div style="font-size:16px;color:#111827;margin-bottom:10px;">
            Dijital dünyada fark yaratmanın tam zamanı! Web tasarımı, e-ticaret, sosyal medya ve Google reklam hizmetlerimizle markanızı büyütüyoruz.
            Hedeflerinize en uygun stratejiyi birlikte belirleyelim.
          </div>
          <div style="background:#f3f4f6;border-radius:12px;padding:14px;">
            <div style="font-weight:700;margin-bottom:6px;">İletişim</div>
            <div>
              <a href="https://sistemsel.com" target="_blank" rel="noopener">www.sistemsel.com</a><br/>
              Tel: <a href="tel:+902128712580">0212 871 25 80</a>
            </div>
          </div>
        </div>
      </div>
      </div>



    <script>
      (function(){
        const tbody = document.getElementById('sct-rows');
        const addBtn = document.getElementById('sct-add');
        if(!tbody || !addBtn) return;

        function bindRemove(btn){
          btn.addEventListener('click', function(){
            const tr = btn.closest('tr');
            if(tr) tr.remove();
          });
        }

        document.querySelectorAll('.sct-remove').forEach(bindRemove);

        addBtn.addEventListener('click', function(){
          const idx = tbody.querySelectorAll('tr').length;
          const tr = document.createElement('tr');
          tr.innerHTML = `
            <td><input type="text" name="<?php echo esc_attr(self::OPTION_KEY); ?>[${idx}][id]" value="" class="regular-text" placeholder="btnTeklifAl" /></td>
            <td><input type="text" name="<?php echo esc_attr(self::OPTION_KEY); ?>[${idx}][name]" value="" class="regular-text" placeholder="Teklif Al" /></td>
            <td><button type="button" class="button sct-remove">Kaldır</button></td>
          `;
          tbody.appendChild(tr);
          bindRemove(tr.querySelector('.sct-remove'));
        });
      })();
    </script>
    <?php
  }
}

new Sistemsel_Click_Tracker();



// =========================
// CSV Export (Admin)
// =========================
add_action('admin_post_sct_export_csv', 'sct_export_csv');

function sct_export_csv() {
  if (!current_user_can('manage_options')) {
    wp_die('Yetkisiz erişim');
  }

  check_admin_referer('sct_export_csv_action');

  global $wpdb;
  $table = $wpdb->prefix . 'sct_clicks';

  $from = isset($_GET['from']) ? sanitize_text_field($_GET['from']) : '';
  $to   = isset($_GET['to']) ? sanitize_text_field($_GET['to']) : '';

  if ($from === '' || $to === '') {
    $to = wp_date('Y-m-d');
    $from = wp_date('Y-m-d', strtotime('-6 days'));
  }

  $rows = $wpdb->get_results(
    $wpdb->prepare(
      "SELECT DATE(clicked_at) as day, element_name, element_id, COUNT(*) as clicks
       FROM $table
       WHERE clicked_at >= %s AND clicked_at <= %s
       GROUP BY DATE(clicked_at), element_id
       ORDER BY day DESC",
      $from . " 00:00:00",
      $to . " 23:59:59"
    ),
    ARRAY_A
  );

  header('Content-Type: text/csv; charset=utf-8');
  header('Content-Disposition: attachment; filename=sistemsel-click-tracker-' . $from . '-to-' . $to . '.csv');

  $output = fopen('php://output', 'w');
  fputcsv($output, ['Gün', 'Element Adı', 'CSS ID', 'Tıklama Sayısı']);

  foreach ($rows as $row) {
    fputcsv($output, [
      $row['day'],
      $row['element_name'],
      $row['element_id'],
      $row['clicks']
    ]);
  }

  fclose($output);
  exit;
}
