<?php
/*
Plugin Name: Sistemsel SMTP Servis
Plugin URI: https://plugin.sistemsel.com/updates/sistemsel-smtp-servis/
Description: WordPress için güvenilir SMTP e-posta gönderimi. Ayarlar + Test e-postası + Bağlantı Testi + Loglar + Export/Import + Servis şablonları.
Version: 2.0.5
Author: Sistemsel Medya
Author URI: https://sistemsel.com
License: GPL2+
Text Domain: sistemsel-smtp-servis
*/

if ( ! defined( 'ABSPATH' ) ) exit;

define( 'SSMTPS_VERSION', '2.0.5' );
define( 'SSMTPS_SLUG', 'sistemsel-smtp-servis' );
define( 'SSMTPS_UPDATER_JSON', 'https://plugin.sistemsel.com/updates/sistemsel-smtp-servis/metadata.json' );
define( 'SSMTPS_PLUGIN_FILE', __FILE__ );
define( 'SSMTPS_PLUGIN_DIR', plugin_dir_path(__FILE__) );
define( 'SSMTPS_PLUGIN_URL', plugin_dir_url(__FILE__) );

require_once SSMTPS_PLUGIN_DIR . 'includes/class-sstmps-updater.php';

/**
 * Option key
 */
function sstmps_option_key() { return 'sstmps_settings'; }

/**
 * Default settings
 */
function sstmps_default_settings() {
    return array(
        'host'          => '',
        'encryption'    => '', // '', ssl, tls
        'port'          => '',
        'autotls'       => 'yes', // yes/no
        'auth'          => 'yes', // yes/no
        'username'      => '',
        'password'      => '',
        'from_name'     => get_bloginfo( 'name' ),
        'from_email'    => get_option( 'admin_email' ),
        'reply_to'      => '',
        'disable_wp_mail_from_override' => 'no',
        'force_override' => 'no',
        'preset'        => '',
    );
}

/**
 * Get settings (merged with defaults)
 */
function sstmps_get_settings() {
    $saved = get_option( sstmps_option_key(), array() );
    return wp_parse_args( $saved, sstmps_default_settings() );
}

/**
 * Create logs table on activation
 */
register_activation_hook( __FILE__, function(){
    global $wpdb;
    $table = $wpdb->prefix . 'sstmps_logs';
    $charset = $wpdb->get_charset_collate();
    $sql = "CREATE TABLE IF NOT EXISTS {$table} (
        id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
        created_at DATETIME NOT NULL,
        to_email TEXT,
        subject TEXT,
        status VARCHAR(20) NOT NULL,
        error TEXT,
        PRIMARY KEY  (id)
    ) {$charset};";
    require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
    dbDelta( $sql );
});

/**
 * Admin menu
 */
add_action( 'admin_menu', function() {
    add_menu_page(
        __( 'SMTP Servis', 'sistemsel-smtp-servis' ),
        __( 'SMTP Servis', 'sistemsel-smtp-servis' ),
        'manage_options',
        'sstmps',
        'sstmps_render_admin_page',
        'dashicons-email',
        81
    );
    add_submenu_page(
        'sstmps',
        __( 'Loglar', 'sistemsel-smtp-servis' ),
        __( 'Loglar', 'sistemsel-smtp-servis' ),
        'manage_options',
        'sstmps-logs',
        'sstmps_render_logs_page'
    );
} );

/**
 * Register settings & assets
 */
add_action( 'admin_init', function() {
    register_setting( 'sstmps_settings_group', sstmps_option_key(), 'sstmps_sanitize_settings' );
    wp_register_style( 'sstmps-admin', SSMTPS_PLUGIN_URL . 'assets/css/admin.css', array(), SSMTPS_VERSION );
    wp_register_script( 'sstmps-admin', SSMTPS_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), SSMTPS_VERSION, true );
} );

/**
 * Ensure assets localized for our pages
 */
add_action('admin_enqueue_scripts', function($hook){
    if (strpos($hook, 'sstmps') === false) return;
    wp_enqueue_style('sstmps-admin');
    wp_enqueue_script('sstmps-admin');
    wp_localize_script('sstmps-admin', 'sstmpsAdmin', array(
        'nonce' => wp_create_nonce('sstmps_conn_test'),
        'ajaxurl' => admin_url('admin-ajax.php')
    ));
});

/**
 * Sanitize
 */
function sstmps_sanitize_settings( $input ) {
    $out = sstmps_get_settings();
    $out['host']       = sanitize_text_field( $input['host'] ?? '' );
    $out['encryption'] = in_array( $input['encryption'] ?? '', array('', 'ssl', 'tls'), true ) ? $input['encryption'] : '';
    $out['port']       = preg_replace( '/[^0-9]/', '', $input['port'] ?? '' );
    $out['autotls']    = ( ($input['autotls'] ?? 'yes') === 'no' ) ? 'no' : 'yes';
    $out['auth']       = ( ($input['auth'] ?? 'yes') === 'no' ) ? 'no' : 'yes';
    $out['username']   = sanitize_text_field( $input['username'] ?? '' );
    if ( isset( $input['password'] ) && $input['password'] !== '' ) $out['password'] = $input['password'];
    $out['from_name']  = sanitize_text_field( $input['from_name'] ?? get_bloginfo('name') );
    $out['from_email'] = sanitize_email( $input['from_email'] ?? get_option('admin_email') );
    $out['reply_to']   = sanitize_email( $input['reply_to'] ?? '' );
    $out['disable_wp_mail_from_override'] = ( ($input['disable_wp_mail_from_override'] ?? 'no') === 'yes' ) ? 'yes' : 'no';
    $out['force_override'] = ( ($input['force_override'] ?? 'no') === 'yes' ) ? 'yes' : 'no';
    $out['preset']     = sanitize_text_field( $input['preset'] ?? '' );
    return $out;
}

/**
 * Admin page renderer
 */
function sstmps_render_admin_page() {
    if ( ! current_user_can( 'manage_options' ) ) return;
    $settings = sstmps_get_settings();
    ?>
    <div class="wrap">
        <h1>📧 Sistemsel SMTP Servis</h1>
        <?php if ( isset($_GET['settings-updated']) ): ?>
            <div class="notice notice-success is-dismissible"><p>Ayarlar kaydedildi.</p></div>
        <?php endif; ?>

        <div class="sstmps-grid">
            <form method="post" action="options.php" class="sstmps-card">
                <?php settings_fields( 'sstmps_settings_group' ); ?>
                <table class="form-table" role="presentation">
                    <tbody>
                        <tr>
                            <th scope="row"><label for="preset">Servis Şablonu</label></th>
                            <td>
                                <select id="preset" name="<?php echo esc_attr( sstmps_option_key() ); ?>[preset]">
                                    <option value=""><?php echo esc_html__('Manuel ayarla','sistemsel-smtp-servis'); ?></option>
                                    <option value="gmail" <?php selected($settings['preset'],'gmail'); ?>>Gmail</option>
                                    <option value="outlook" <?php selected($settings['preset'],'outlook'); ?>>Outlook</option>
                                    <option value="yandex" <?php selected($settings['preset'],'yandex'); ?>>Yandex</option>
									<option value="hositnger" <?php selected($settings['preset'],'hositnger'); ?>>Hostinger</option>
                                    <option value="zoho" <?php selected($settings['preset'],'zoho'); ?>>Zoho</option>
                                    <option value="sendgrid" <?php selected($settings['preset'],'sendgrid'); ?>>SendGrid (SMTP)</option>
                                </select>
                                <p class="description">Şablon seçmezsen, aşağıdaki alanları manuel doldur. Boş bırakmak <strong>hata değildir</strong>.</p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="host">SMTP Host</label></th>
                            <td><input type="text" id="host" name="<?php echo esc_attr( sstmps_option_key() ); ?>[host]" value="<?php echo esc_attr($settings['host']); ?>" class="regular-text" required></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="encryption">Şifreleme</label></th>
                            <td>
                                <select id="encryption" name="<?php echo esc_attr( sstmps_option_key() ); ?>[encryption]">
                                    <option value="" <?php selected($settings['encryption'], ''); ?>>Yok</option>
                                    <option value="ssl" <?php selected($settings['encryption'], 'ssl'); ?>>SSL</option>
                                    <option value="tls" <?php selected($settings['encryption'], 'tls'); ?>>TLS</option>
                                </select>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="port">SMTP Bağlantı Noktası</label></th>
                            <td><input type="text" id="port" name="<?php echo esc_attr( sstmps_option_key() ); ?>[port]" value="<?php echo esc_attr($settings['port']); ?>" class="small-text" placeholder="25, 465 veya 587"></td>
                        </tr>
                        <tr>
                            <th scope="row">Otomatik TLS</th>
                            <td>
                                <label><input type="radio" name="<?php echo esc_attr( sstmps_option_key() ); ?>[autotls]" value="yes" <?php checked($settings['autotls'], 'yes'); ?>> Evet</label>
                                &nbsp;
                                <label><input type="radio" name="<?php echo esc_attr( sstmps_option_key() ); ?>[autotls]" value="no" <?php checked($settings['autotls'], 'no'); ?>> Hayır</label>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Kimlik Doğrulama</th>
                            <td>
                                <label><input type="radio" name="<?php echo esc_attr( sstmps_option_key() ); ?>[auth]" value="yes" <?php checked($settings['auth'], 'yes'); ?>> Evet</label>
                                &nbsp;
                                <label><input type="radio" name="<?php echo esc_attr( sstmps_option_key() ); ?>[auth]" value="no" <?php checked($settings['auth'], 'no'); ?>> Hayır</label>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="username">SMTP Kullanıcı Adı</label></th>
                            <td><input type="text" id="username" name="<?php echo esc_attr( sstmps_option_key() ); ?>[username]" value="<?php echo esc_attr($settings['username']); ?>" class="regular-text"></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="password">SMTP Şifresi</label></th>
                            <td><input type="password" id="password" name="<?php echo esc_attr( sstmps_option_key() ); ?>[password]" value="" class="regular-text" placeholder="(değiştirmeyecekseniz boş bırakın)"></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="from_name">Gönderen Adı</label></th>
                            <td><input type="text" id="from_name" name="<?php echo esc_attr( sstmps_option_key() ); ?>[from_name]" value="<?php echo esc_attr($settings['from_name']); ?>" class="regular-text"></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="from_email">Gönderen E-Posta</label></th>
                            <td><input type="email" id="from_email" name="<?php echo esc_attr( sstmps_option_key() ); ?>[from_email]" value="<?php echo esc_attr($settings['from_email']); ?>" class="regular-text"></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="reply_to">Yanıt Adresi (Opsiyonel)</label></th>
                            <td><input type="email" id="reply_to" name="<?php echo esc_attr( sstmps_option_key() ); ?>[reply_to]" value="<?php echo esc_attr($settings['reply_to']); ?>" class="regular-text"></td>
                        </tr>
                        <tr>
                            <th scope="row">Gelişmiş</th>
                            <td>
                                <label style="display:block;margin-bottom:6px;">
                                    <input type="checkbox" name="<?php echo esc_attr( sstmps_option_key() ); ?>[disable_wp_mail_from_override]" value="yes" <?php checked($settings['disable_wp_mail_from_override'], 'yes'); ?>>
                                    WP 'from' filtrelerini değiştirme (Gerekmedikçe işaretlemeyin)
                                </label>
                                <label>
                                    <input type="checkbox" name="<?php echo esc_attr( sstmps_option_key() ); ?>[force_override]" value="yes" <?php checked($settings['force_override'], 'yes'); ?>>
                                    Diğer e-posta eklentileri ayarlasa bile bu SMTP ayarlarını zorla
                                </label>
                            </td>
                        </tr>
                    </tbody>
                </table>
                <p>
                    <button type="button" class="button" id="sstmps-conn-test">Bağlantıyı Test Et</button>
                    <span id="sstmps-conn-result" style="margin-left:8px;"></span>
                </p>
                <?php submit_button(); ?>
            </form>

            <div class="sstmps-card">
                <div style="text-align:center;margin-bottom:12px;">
                    <img src="https://plugin.sistemsel.com/sistemsel/logo.png" alt="Sistemsel" style="height:48px;">
                </div>
                <p style="margin:0 0 10px;line-height:1.6;">
                    Dijital dünyada fark yaratmanın tam zamanı! Web tasarımı, e-ticaret, sosyal medya ve Google reklam hizmetlerimizle markanızı büyütüyoruz. Hedeflerinize en uygun stratejiyi birlikte belirleyelim.
                </p>
                <div style="margin-top:10px;padding:12px;background:#f7f7f7;border-radius:8px;">
                    <strong>İletişim</strong><br>
                    <a href="https://www.sistemsel.com" target="_blank" rel="noopener">www.sistemsel.com</a><br>
                    Tel: 0212 871 25 80
                </div>
                <hr>
                <form method="post">
                    <?php wp_nonce_field( 'sstmps_export', 'sstmps_export_nonce' ); ?>
                    <button class="button" name="sstmps_export" value="1">Ayarları Dışa Aktar (JSON)</button>
                </form>
                <form method="post" enctype="multipart/form-data" style="margin-top:8px;">
                    <?php wp_nonce_field( 'sstmps_import', 'sstmps_import_nonce' ); ?>
                    <input type="file" name="sstmps_import_file" accept="application/json">
                    <button class="button" name="sstmps_import" value="1">İçe Aktar</button>
                </form>
            </div>
        </div>

        <hr>
        <h2>Test E-Postası Gönder</h2>
        <form method="post">
            <?php wp_nonce_field( 'sstmps_send_test', 'sstmps_test_nonce' ); ?>
            <p>
                <label for="sstmps_test_to">Alıcı E-Posta:</label>
                <input type="email" id="sstmps_test_to" name="sstmps_test_to" class="regular-text" required>
                <?php submit_button( 'Test E-Postası Gönder', 'secondary', 'sstmps_send_test', false ); ?>
            </p>
        </form>
        <?php
            if ( isset($_POST['sstmps_send_test']) && check_admin_referer( 'sstmps_send_test', 'sstmps_test_nonce' ) ) {
                $to = sanitize_email( $_POST['sstmps_test_to'] );
                $subject = 'Sistemsel SMTP Servis Test E-Postası';
                ob_start();
                ?>
                <div style="max-width:600px;margin:0 auto;font-family:Arial,Helvetica,sans-serif;">
                    <div style="padding:16px;border:1px solid #eaeaea;border-radius:8px">
                        <div style="text-align:center;margin-bottom:16px;">
                            <img src="https://plugin.sistemsel.com/sistemsel/logo.png" alt="Sistemsel" style="height:48px;">
                        </div>
                        <h1 style="font-size:20px;margin:0 0 12px;">Merhaba 👋</h1>
                        <p style="margin:0 0 12px;line-height:1.6;">
                            Bu mesaj <strong>Sistemsel SMTP Servis</strong> eklentiniz aracılığıyla gönderilen bir test e-postasıdır.
                        </p>
                        <p style="margin:0 0 12px;line-height:1.6;">
                            Dijital dünyada fark yaratmanın tam zamanı! Web tasarımı, e-ticaret, sosyal medya ve Google reklam hizmetlerimizle markanızı büyütüyoruz. Hedeflerinize en uygun stratejiyi birlikte belirleyelim.
                        </p>
                        <div style="margin:20px 0;padding:12px;background:#f7f7f7;border-radius:8px;">
                            <strong>İletişim</strong><br>
                            <a href="https://www.sistemsel.com" target="_blank" rel="noopener">www.sistemsel.com</a><br>
                            Tel: 0212 871 25 80
                        </div>
                        <p style="color:#999;font-size:12px;">Gönderim zamanı: <?php echo esc_html( current_time('mysql') ); ?></p>
                    </div>
                </div>
                <?php
                $message = ob_get_clean();
                $headers = array('Content-Type: text/html; charset=UTF-8');
                if ( ! empty( sstmps_get_settings()['reply_to'] ) ) {
                    $headers[] = 'Reply-To: ' . sstmps_get_settings()['reply_to'];
                }
                $sent = wp_mail( $to, $subject, $message, $headers );

                // Logla
                global $wpdb;
                $table = $wpdb->prefix . 'sstmps_logs';
                if ( $sent ) {
                    $wpdb->insert( $table, array(
                        'created_at' => current_time('mysql'),
                        'to_email'   => $to,
                        'subject'    => $subject,
                        'status'     => 'success',
                        'error'      => ''
                    ) );
                    echo '<div class="notice notice-success is-dismissible"><p>Test e-postası <strong>' . esc_html($to) . '</strong> adresine gönderildi.</p></div>';
                } else {
                    $wpdb->insert( $table, array(
                        'created_at' => current_time('mysql'),
                        'to_email'   => $to,
                        'subject'    => $subject,
                        'status'     => 'failed',
                        'error'      => 'Gönderim başarısız'
                    ) );
                    echo '<div class="notice notice-error"><p>Test e-postası gönderilemedi. Lütfen SMTP ayarlarını kontrol edin.</p></div>';
                }
            }
        ?>
        <p style="margin-top:24px;opacity:.7">Sürüm: 2.0.5</p>
    </div>
    <?php
}

/**
 * Logs page
 */
function sstmps_render_logs_page() {
    if ( ! current_user_can( 'manage_options' ) ) return;
    global $wpdb;
    $table = $wpdb->prefix . 'sstmps_logs';

    if ( isset($_POST['sstmps_clear_logs']) && check_admin_referer('sstmps_clear_logs') ) {
        $wpdb->query( "TRUNCATE TABLE {$table}" );
        echo '<div class="notice notice-success is-dismissible"><p>Loglar temizlendi.</p></div>';
    }

    $items = $wpdb->get_results( "SELECT * FROM {$table} ORDER BY id DESC LIMIT 100" );
    ?>
    <div class="wrap">
        <h1>SMTP Logları</h1>
        <form method="post" style="margin:10px 0;">
            <?php wp_nonce_field('sstmps_clear_logs'); ?>
            <button class="button button-secondary" name="sstmps_clear_logs" value="1" onclick="return confirm('Logları temizlemek istediğinize emin misiniz?')">Logları Temizle</button>
        </form>
        <table class="widefat striped">
            <thead><tr><th>ID</th><th>Tarih</th><th>Alıcı</th><th>Konu</th><th>Durum</th><th>Hata</th></tr></thead>
            <tbody>
            <?php if ( $items ): foreach ( $items as $r ): ?>
                <tr>
                    <td><?php echo (int)$r->id; ?></td>
                    <td><?php echo esc_html($r->created_at); ?></td>
                    <td><?php echo esc_html($r->to_email); ?></td>
                    <td><?php echo esc_html($r->subject); ?></td>
                    <td><?php echo esc_html($r->status); ?></td>
                    <td><?php echo esc_html($r->error); ?></td>
                </tr>
            <?php endforeach; else: ?>
                <tr><td colspan="6">Kayıt yok.</td></tr>
            <?php endif; ?>
            </tbody>
        </table>
    </div>
    <?php
}

/**
 * Export / Import handlers
 */
add_action( 'admin_init', function() {
    if ( isset($_POST['sstmps_export']) && check_admin_referer('sstmps_export','sstmps_export_nonce') ) {
        $settings = sstmps_get_settings();
        header('Content-Type: application/json; charset=utf-8');
        header('Content-Disposition: attachment; filename="sstmps-settings.json"');
        echo wp_json_encode($settings);
        exit;
    }
    if ( isset($_POST['sstmps_import']) && check_admin_referer('sstmps_import','sstmps_import_nonce') ) {
        if ( ! empty($_FILES['sstmps_import_file']['tmp_name']) ) {
            $json = file_get_contents( $_FILES['sstmps_import_file']['tmp_name'] );
            $data = json_decode( $json, true );
            if ( is_array($data) ) {
                update_option( sstmps_option_key(), wp_parse_args($data, sstmps_default_settings()) );
                wp_safe_redirect( add_query_arg( array('page'=>'sstmps','settings-updated'=>'true'), admin_url('admin.php') ) );
                exit;
            }
        }
    }
} );

/**
 * Capture wp_mail_failed errors into logs
 */
add_action( 'wp_mail_failed', function( $wp_error ) {
    global $wpdb;
    $table = $wpdb->prefix . 'sstmps_logs';
    $data = $wp_error->get_error_data();
    $to = isset($data['to']) ? (is_array($data['to']) ? implode(',', $data['to']) : $data['to']) : '';
    $subj = isset($data['subject']) ? $data['subject'] : '';
    $wpdb->insert( $table, array(
        'created_at' => current_time('mysql'),
        'to_email'   => $to,
        'subject'    => $subj,
        'status'     => 'failed',
        'error'      => $wp_error->get_error_message()
    ) );
}, 10, 1 );

/**
 * AJAX: Connection test
 */
add_action( 'wp_ajax_sstmps_conn_test', function(){
    if ( ! current_user_can('manage_options') ) wp_send_json_error('Yetkisiz.');
    check_ajax_referer( 'sstmps_conn_test', 'nonce' );
    $s = sstmps_get_settings();

    if ( empty($s['host']) ) wp_send_json_error('SMTP host boş olamaz.');

    if ( ! class_exists('\\PHPMailer\\PHPMailer\\PHPMailer') ) {
        require_once ABSPATH . WPINC . '/PHPMailer/PHPMailer.php';
        require_once ABSPATH . WPINC . '/PHPMailer/SMTP.php';
        require_once ABSPATH . WPINC . '/PHPMailer/Exception.php';
    }

    $smtp = new \PHPMailer\PHPMailer\SMTP();
    $smtp->do_debug = 0;
    $secure = $s['encryption'];
    $port = (int) ( $s['port'] ?: ( $secure==='ssl' ? 465 : ( $secure==='tls' ? 587 : 25 ) ) );
    $host = $s['host'];

    $connected = false;
    $error = '';

    try {
        $connected = @$smtp->connect( $host, $port, 10 );
        if ( ! $connected ) {
            $error = 'Sunucuya bağlanılamadı.';
        } else {
            if ( $secure === 'tls' ) {
                if ( ! $smtp->startTLS() ) { $connected = false; $error = 'STARTTLS başarısız.'; }
            }
            if ( $connected && $s['auth'] === 'yes' ) {
                if ( ! $smtp->hello(get_bloginfo('name')) ) { $connected = false; $error = 'EHLO hatası.'; }
                if ( ! $smtp->authenticate( $s['username'], $s['password'] ) ) { $connected = false; $error = 'Kimlik doğrulama başarısız.'; }
            }
        }
    } catch ( \Exception $e ) {
        $connected = false;
        $error = $e->getMessage();
    } finally {
        if ( $smtp->connected() ) $smtp->quit(true);
    }

    if ( $connected ) {
        wp_send_json_success( array('message' => 'Bağlantı başarılı (host/port erişilebilir).') );
    } else {
        $msg = !empty($error) ? $error : 'Bağlantı başarısız. Host/port/encryption bilgilerini kontrol edin.';
        wp_send_json_error( array('message' => $msg) );
    }
} );

/**
 * PHPMailer Configure
 */
function sstmps_configure_phpmailer( $phpmailer ) {
    $s = sstmps_get_settings();
    if ( empty( $s['host'] ) ) return;

    $phpmailer->isSMTP();
    $phpmailer->Host       = $s['host'];
    $phpmailer->Port       = (int) ( $s['port'] ?: 0 );
    $phpmailer->SMTPAuth   = ( $s['auth'] === 'yes' );
    $phpmailer->SMTPSecure = $s['encryption']; // '', 'ssl', 'tls'
    $phpmailer->SMTPAutoTLS= ( $s['autotls'] === 'yes' );

    if ( $phpmailer->SMTPAuth ) {
        $phpmailer->Username = $s['username'];
        $phpmailer->Password = $s['password'];
    }

    $from_email = is_email( $s['from_email'] ) ? $s['from_email'] : get_option('admin_email');
    $from_name  = $s['from_name'] ?: get_bloginfo('name');

    if ( $s['disable_wp_mail_from_override'] !== 'yes' ) {
        add_filter( 'wp_mail_from', function() use ( $from_email ) { return $from_email; } );
        add_filter( 'wp_mail_from_name', function() use ( $from_name ) { return $from_name; } );
    }

    try { $phpmailer->setFrom( $from_email, $from_name, false ); } catch ( \PHPMailer\PHPMailer\Exception $e ) {}
    if ( ! empty( $s['reply_to'] ) && is_email( $s['reply_to'] ) ) {
        try { $phpmailer->addReplyTo( $s['reply_to'] ); } catch ( \PHPMailer\PHPMailer\Exception $e ) {}
    }
}
add_action( 'phpmailer_init', 'sstmps_configure_phpmailer', 5 );
add_action( 'phpmailer_init', function($phpmailer){
    $s = sstmps_get_settings();
    if ( isset($s['force_override']) && $s['force_override'] === 'yes' ) {
        sstmps_configure_phpmailer( $phpmailer );
    }
}, 999 );

/**
 * Initialize updater
 */
add_action( 'init', function() {
    if ( is_admin() ) {
        new SSTMPS_Updater( SSMTPS_PLUGIN_FILE, SSMTPS_UPDATER_JSON, SSMTPS_VERSION );
    }
} );
